breeds [
  reactants      ;; reactant molecules (NO2), shown as green
  products       ;; product molecules (N2O4), shown as red
]

turtles-own [
  reacting?      ;; used to ensure that reactants can only participate in one reaction at a time
]

globals [
  Kf                 ;; when temp-effect is on, replaces Kb
  Kr                 ;; when temp-effect is on, replaces Ku
  Keq                ;; equilibrium constant
  react-conc         ;; concentration of reactant
  prod-conc          ;; concentration of product
  black-patches      ;; agentset of all the patches inside the reaction container
]

to setup
  ca
  set-default-shape reactants "NO2"
  set-default-shape products "N2O4"
  draw-box                                        ;; draws the reaction container
  create-custom-reactants 200                     ;; generates reactants
    [ set color green
      set reacting? false
      randomize ]                                 ;; distributes reactants across the screen
  update-concentrations
  do-plotting
end

to update-concentrations
  set react-conc (count reactants / count black-patches) * 100
  set prod-conc (count products / count black-patches) * 100
end

to go
  ifelse temp-effect?
    ;; if temp-effect is ON, Kf  replaces Kb and Kr replaces Ku
    ;; Kf and Kr are calculated with the Arrhenius Equation, k = ln A - Ea / RT .
    ;; ln A, Ea, and R are all constants for a reaction. Below, the constants have simulated 
    ;; values that were chosen to produce the same qualitative results as the real-world
    ;; reaction. Because the number of molecules in the model is so small, we use simulated
    ;; constants to keep the values of each variable in a useful range for the model.
    ;; The important thing to note is how each K value varies according to the
    ;; temperature of the reaction.
    [ set Kf (5 + (3000 /(8 * (temp + 273))))
      set Kr (15 - (8000 / (8 * (temp + 273)))) ]
    ;; if temp-effect is OFF, set Keq based on concentrations
    [ set Keq (prod-conc / (react-conc ^ 2)) * 100 ]
  ask turtles
    [ bounce                        ;; bounce off walls
      fd 1                          ;; wander aimlessly
      rt random-float 10 - random-float 10 ]
  ask reactants
    [ react-forward ]
  ask products
    [ react-backward ]
  update-concentrations
  do-plotting
end

;; see explanations of "react-forward" and "react-backward" in the info window

to react-forward  ;; turtle procedure
  let chance 0
  ifelse temp-effect?
    [ set chance Kf ]   ;; multiply Kb by 2 because 2 molecules are involved
    [ set chance Kb ]  ;; use of 1.0 here is arbitrary because calculations for real Kb would be too difficult
  if (any? other-reactants-here) and (random-float 10.0 < chance)
    [ set reacting? true  ;; prevent anyone else from trying to react with us
      let partner (random-one-of other-reactants-here)
      if not reacting?-of partner
        [ set reacting?-of partner true  ;; prevent anyone else from trying to react with our partner
          set breed-of partner products
          set reacting?-of partner false
          set color-of partner red
          die ]
      set reacting? false ]
end

to react-backward  ;; turtle procedure
  let chance Ku
  if temp-effect?
    [ set chance Kr ]
  if (random-float 1000.0 < chance)
    [ set breed reactants
      set color green
      hatch 1
        [ rt 180 ] ]
end

;; turtle procedure to bounce molecules off the yellow walls
to bounce
  let box-edge (screen-edge-x - edge-size)  ;; setting this first makes the calculations a bit simpler
  ; check: hitting top or bottom wall?
  if ((pycor = box-edge)       and ((heading > 270) or (heading < 90))) or
     ((pycor = (- box-edge)) and ((heading > 90) and (heading < 270)))
    ; if so, reflect heading around y axis
    [ set heading (180 - heading) ]
  ; check: hitting left or right wall?
  if ((pxcor = box-edge)       and ((heading > 0) and (heading < 180))) or
     ((pxcor = (- box-edge)) and ((heading > 180)))
    ; if so, reflect heading around x axis
    [ set heading (- heading) ]
end

;;observer procedures to add more molecules to the model
to add-reactant
  create-custom-reactants 20
    [ set color green
      set reacting? false
      randomize ]
end

to add-product
  create-custom-products 20
    [ set reacting? false
      set color red
      randomize ]
end

;;;; plotting procedures

to do-plotting
  set-current-plot "Concentrations"
  set-current-plot-pen "Reactant"
  plot react-conc
  set-current-plot-pen "Product"
  plot prod-conc
end

to draw-box
  ask patches
    [ ifelse (abs pxcor > (screen-edge-x - edge-size)) or
             (abs pycor > (screen-edge-y - edge-size))
        [ set pcolor yellow ]
        [ set pcolor black ] ]
  set black-patches patches with [pcolor = black]
  ask turtles
    [ randomize ]
end

;; turtle procedure that places turtles randomly within the box
to randomize
  let random-patch random-one-of black-patches
  setxy (pxcor-of random-patch) (pycor-of random-patch)
  rt random-float 360
end


; *** NetLogo Model Copyright Notice ***
;
; This model was created as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN
; CLASSROOMS.  The project gratefully acknowledges the support of the
; National Science Foundation (REPP program) -- grant number REC #9814682.
;
; Copyright 2001 by Uri Wilensky.  Updated 2002.  All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2001).  NetLogo Simple Kinetics 2 model.
; http://ccl.northwestern.edu/netlogo/models/SimpleKinetics2.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 1998 by Uri Wilensky.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/SimpleKinetics2
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
255
10
640
416
12
12
15.0
1
10
1
1
1
0

CC-WINDOW
5
488
649
583
Command Center

BUTTON
4
21
79
54
setup
setup
NIL
1
T
OBSERVER
T
NIL

BUTTON
84
21
153
54
NIL
go
T
1
T
OBSERVER
T
NIL

BUTTON
5
228
101
261
redraw box
draw-box
NIL
1
T
OBSERVER
T
NIL

SLIDER
5
194
101
227
edge-size
edge-size
1
10
1
1
1
NIL

SWITCH
116
194
251
227
temp-effect?
temp-effect?
1
1
-1000

SLIDER
3
86
160
119
Kb
Kb
0.0
10.0
10.0
0.1
1
NIL

SLIDER
3
120
160
153
Ku
Ku
0.0
100.0
34.0
1.0
1
NIL

BUTTON
169
120
252
153
add reactant
add-reactant
NIL
1
T
OBSERVER
T
NIL

BUTTON
169
86
252
119
add product
add-product
NIL
1
T
OBSERVER
T
NIL

SLIDER
116
228
251
261
temp
temp
-200.0
200.0
200.0
1.0
1
deg C

MONITOR
13
269
122
318
Reactant Conc
react-conc
1
1

MONITOR
123
269
226
318
Product Conc
prod-conc
1
1

PLOT
4
319
247
474
Concentrations
time
# molecules
0.0
10.0
0.0
10.0
true
true
PENS
"Reactant" 1.0 0 -11352576 true
"Product" 1.0 0 -65536 true

TEXTBOX
5
164
102
193
Volume (pause model to change)

TEXTBOX
165
68
255
86
Concentration

TEXTBOX
117
176
207
194
Temperature

TEXTBOX
4
68
161
86
Rates (when temp-effect off)

@#$#@#$#@
WHAT IS IT?
-----------
This model demonstrates the properties of LeChatelier's Principle. This chemical principle states that if a system that is at equilibrium is perturbed, the system will readjust to establish a new equilibrium. For example, if you add reactants to a reversible reaction that is at equilibrium, the system will shift to generate more products and establish a new equilibrium. The principle can also be described with chemical equations.

Below is a generic equation which depicts two molecules of reactant A combining to form one molecule or product B. The reaction is reversible, meaning that the one molecule of B can break down into two molecules of A.

|                  Kb
|        A + A <=======> B
|                  Ku

An example of such a reaction would be dimerization of the gas nitrous oxide:

|                  Kb
|        2 NO  <=======> N O
|            2     Ku     2 4

This reaction is an example of a complex reaction which consists of two elementary reactions.  The forward bimolecular reaction

|                 Kb
|        A + A --------> B

is characterized by the constant Kb and the reverse unimolecular reaction

|              Ku
|        B ---------> A + A

The equilibrium rate constant for the entire reaction (Keq) is equal to [B] / [A] ^ 2. Each of the rate constants in the equations above has units of s^-1. They are empirically derived constants that when combined with the reaction concentrations tell you how fast the reaction proceeds according to the reaction rate law. The rate law ultimately tells you how many Molar units of a molecule react per second. For the reaction above the forward rate law is RATE = Kb[A]^2 and the reverse rate law is RATE = Ku[B].

Note that because we are simulating the reaction, the values of Kb and Ku in this reaction are not real-world values.  It would be necessary to use several differential equations to calculate the real values of Kb, Ku and Keq, however, several qualitative features of their relationships can be seen using this model. Reaction equilibrium is reached when a system reaches a steady-state. This is not to say that reactions have stopped occurring!  Microscopic changes in equilibrium still take place, but to our eyes and our measurements the system appears stable because the forward and reverse rates are equal.

The rate at which a reaction reaches equilibrium as well as the state of the equilibrium system both depend upon the rate constants, the temperature, the concentration of reactants and products and, when a gas is involved, the volume of the container. When a system has reached equilibrium, changes to any of the variables above result in a change in the system to establish a new equilibrium. This effect is predicted using LeChatelier's Principle. We can use our model to discover the role of each variable (temperature, volume, concentration and rate constant) in LeChatelier's Principle.


HOW TO USE IT
--------------
To start off:

Choose the values of Kb and Ku with appropriate sliders:
- Kb controls the rate of the forward reaction by which two green molecules turn bimolecularly into a single red molecule.
- Ku controls the rate of the reverse reaction, by which a red molecule turns unimolecularly into two green molecules.

Having chosen appropriate values of the constants, press SETUP to clear the screen and create an initial number of green molecules.  Note: we do not create red molecules initially, although this can be done in principal.

Press RUN to start the simulation.

Set the size of the yellow box using the EDGE-SIZE slider. (If you would like to change the size while you are running a model. Press RUN to stop the model, adjust the EDGE-SIZE slider and redraw the box using the REDRAW BOX button. Resume the reaction by pressing RUN.)

After viewing the effects of several different rate constant values, use the other sliders and buttons to observe how concentration, volume, and temperature affect the equilibrium.

A note on the temperature variable. Temperature changes have a unique effect on equilibrium compared with the other variables. You can observe this effect by toggling the TEMP-EFFECT button on or off and using the slider to set the temperature of the reaction in centigrade.


THINGS TO NOTICE
----------------
You will see molecules wandering around the screen and changing color.  Pay more attention to the plot of the concentrations.  Do the plots soon reach stationary concentrations?

How does changing the concentrations of reactants and products in the system affect the equilibrium? Does it take more or less time to reach a stationary condition under various conditions?

What is the effect of temperature on the equilibrium of the system compared to volume or concentration?  In the Procedures window, note how rate constants are calculated based on the temperature.

Notice how the ratio of products to reactants changes with changes to the system. Does the ratio change much with each factor? Make a window that show the value of Keq to help you determine this.

Why do the traces of each breed eventual balance around a constant average? How come this value is an average and not a constant?


THINGS TO TRY
-------------
How do the stationary concentrations depend on the values of Kb and Ku?   You can change Ku and Kb while the model is running.   See if you can predict what the stationary concentrations will be with various combinations of Kb and Ku.

Without adding additional reactants or products and with the temperature effect in the off position, note that more red product molecules accumulate when the volume decreases.  Can you explain why?

Observe the progress of the reaction at high and low temperatures.  Does this observed trend fit your expectations?

Try adding some molecules to the system that have no "breed", as an inert gas. Does this affect the equilibrium? Why or why not?


EXTENDING THE MODEL
------------------
Try altering the code so that when two green molecules collide, they produce two red molecules instead of one. Likewise, alter it so that two red molecules must collide to form two green molecules. Observe the effect of volume on this system. Is the effect as you predicted?

What would the effect of adding a catalyst to the system be?  Add a catalyst breed that accelerates the reaction and observe the trend.  Are you surprised?

Add a monitor that measures the equilibrium constant for the system.  Is it really a constant?


RELATED MODELS
--------------
Simple Kinetics 1, Simple Kinetics 3.  Note that the original StarLogoT version of this model was named "Le Chatelier 1".


NETLOGO FEATURES
----------------
Notice the use of breeds in the model.

Notice how we store an agentset of patches in the BLACK-PATCHES variable.  Computing this agentset once ahead of time (at the time the box is drawn) is faster than recomputing it at every iteration.

It is a little tricky to ensure that a reactant never participates in two reactions simultaneously.  The turtle variable REACTING? is used to control this.  In the future, we will add a new primitive to NetLogo to make this easier.


CREDITS AND REFERENCES
----------------------
Thanks to Mike Stieff for his work on this model.

To refer to this model in academic publications, please use: Wilensky, U. (2001).  NetLogo Simple Kinetics 2 model. http://ccl.northwestern.edu/netlogo/models/SimpleKinetics2. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 2001 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/SimpleKinetics2 for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

n2o4
true
0
Circle -7566196 true true 61 107 88
Circle -7566196 true true 151 106 88
Circle -1 true false 20 64 64
Circle -1 true false 217 63 64
Circle -1 true false 216 171 63
Circle -1 true false 21 173 61

no2
true
0
Circle -7566196 true true 104 104 91
Circle -1 true false 194 119 63
Circle -1 true false 46 120 58

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
